<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class FelInvoice_model extends CI_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    /*
     * ==================================================
     * FLUJO REAL CORRECTO
     * ==================================================
     * 1️⃣ obtener venta
     * 2️⃣ obtener token FEL (certificar)
     * 3️⃣ obtener token NUC (consultar NIT)
     * 4️⃣ consultar NIT
     * 5️⃣ construir JSON
     * 6️⃣ certificar
     * 7️⃣ guardar
     * NUNCA certificar antes del NIT
     * ==================================================
     */
    public function emitirDesdeSale($reference_no, $nit, $tipo = 'FACT', $modo_prueba = true, $taxid = null, $username = null)
    {
        /*
         * ===============================
         * 1️⃣ OBTENER VENTA
         * ===============================
         */
        $sale = $this->db
            ->where('reference_no', $reference_no)
            ->get('sma_sales')
            ->row_array();

        if (!$sale) {
            throw new Exception('Venta no encontrada');
        }

        $items = $this->db
            ->where('sale_id', $sale['id'])
            ->get('sma_sale_items')
            ->result_array();

        /*
         * ===============================
         * 2️⃣ CONEXIÓN CERTIFICADOR
         * ===============================
         */
        $this->load->admin_model('Certificadores_model', 'cert');

        $conexion = $this->db
            ->where('sucursal_id', $sale['warehouse_id'])
            ->where('activo', 1)
            ->get('sma_clientes_certificador_conexion')
            ->row();

        if (!$conexion) {
            throw new Exception('No hay certificador activo');
        }

        $username    = $conexion->usuario;
        $taxid       = $conexion->nit;
        $certificador = $this->cert->get_certificador($conexion->certificador_id);

        /*
         * ===============================
         * 3️⃣ TOKENS (AMBOS)
         * ===============================
         */
        $tokenFEL = $this->cert->get_token_valido($certificador, $conexion);

        log_message('error', 'TOKEN FEL: ' . $tokenFEL);

        if (!$tokenFEL) {
            throw new Exception('No se pudo obtener token Digifact FEL');
        }

        /*
         * ===============================
         * 4️⃣ CONSULTAR NIT (ANTES DE TODO)
         * ===============================
         */
        if (strtoupper($nit) === 'CF') {
            // Obtener datos del cliente de la base de datos
            $cliente_db = $this->db
                ->select('vat_no, name, address, city as municipio, state as departamento, country as pais, postal_code')
                ->where('id', $sale['customer_id'])
                ->where('group_name', 'customer')
                ->get('sma_companies')
                ->row_array();

            if (!$cliente_db) {
                throw new Exception('Cliente CF no encontrado en la base de datos');
            }

            $cliente = $cliente_db; // Asignar directamente el array $cliente_db a $cliente
            $cliente['vat_no'] = 'CF'; // Establecer el NIT como CF
        } else {
            $tokenNUC = $this->cert->get_token_valido_nuc($certificador, $conexion);

            if (!$tokenNUC) {
                throw new Exception('No se pudo obtener token Digifact NUC');
            }

            $cliente = $this->consultarNitDigifact($nit, $tokenNUC, $username, $taxid);
            log_message('error', 'CLIENTE OBTENIDO: ' . json_encode($cliente));
        }

        /*
         * ===============================
         * 5️⃣ JSON
         * ===============================
         */
        // Obtener datos del emisor (biller)
        $biller = $this->db
            ->where('group_name', 'biller')
            ->get('sma_companies')
            ->row_array();

         if (!$biller) {
             throw new Exception('Datos del emisor (biller) no encontrados en la base de datos');
         }

        $json = $this->buildJsonDigifact($sale, $items, $cliente, $biller, $tipo);

        /*
         * ===============================
         * 6️⃣ CERTIFICAR
         * ===============================
         */
        if ($modo_prueba) {
            // Devolver el JSON y los datos del cliente para mostrar en el modal
            return [
                'cliente' => $cliente,
                'json' => json_encode($json, JSON_PRETTY_PRINT), // Devolver el JSON formateado
                'reference_no' => $reference_no,
                'taxid' => $taxid,
                'username' => $username,
            ];
        } else {
            $response = $this->sendDigifact($json, $tokenFEL, $taxid, $username);

            if (empty($response['uuid'])) {
                throw new Exception('Certificador no devolvió UUID');
            }

            /*
             * ===============================
             * 7️⃣ GUARDAR
             * ===============================
             */
           $this->saveInvoice($sale, $cliente, $response, $tipo, $biller);

            return [
                'cliente' => $cliente,
                'uuid' => $response['uuid'] ?? '',
                'ticket' => $reference_no,
            ];
        }
    }

    /*
     * ==================================================
     * CONSULTA NIT (100% IGUAL A DOC DIGIFACT)
     * ==================================================
     */
   private function consultarNitDigifact($nit, $token, $username, $taxid)
{
    $url = "https://nucgt.digifact.com/gt.com.apinuc/api/Shared"
        . "?TAXID=" . $taxid
        . "&DATA1=SHARED_GETINFONITcom"
        . "&DATA2=NIT|" . $nit
        . "&COUNTRY=GT"
        . "&USERNAME=" . $username;

    log_message('error', 'INFONIT URL: ' . $url);

    $headers = [
        "Authorization: " . $token, // SIN Bearer (NUC NO USA BEARER)
        "Content-Type: application/json",
    ];

    $ch = curl_init($url);

    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_HTTPHEADER => $headers,
    ]);

    $result = curl_exec($ch);

    if ($result === false) {
        log_message('error', 'CURL INFONIT: ' . curl_error($ch));
    }

    curl_close($ch);

    log_message('error', 'INFONIT RESPONSE: ' . $result);

    $data = json_decode($result, true);

    if (empty($data['RESPONSE'])) {
        throw new Exception('NIT no encontrado en SAT');
    }

    if ( ! is_array($data['RESPONSE']) || count($data['RESPONSE']) == 0) {
        throw new Exception('NIT no encontrado en el array de respuesta del SAT');
    }

    $r = $data['RESPONSE'][0]; // Accede al primer elemento del array RESPONSE

    // Usar el nombre tal cual lo devuelve la API
    $nombre = $r['NOMBRE'];

    return [
        'vat_no'  => $nit,
        'name'    => $nombre,
        'address' => $r['Direccion'],
        'municipio' => $r['MUNICIPIO'],
        'departamento' => $r['DEPARTAMENTO'],
        'pais' => 'GT',
        'postal_code' => '01001'
    ];
}

    /*
     * ==================================================
     * CERTIFICAR FEL
     * ==================================================
     */
    private function sendDigifact($json, $token, $taxid, $username)
    {
        $this->logJson($json); 

        $url = 'https://felgtaws.digifact.com.gt/gt.com.apinuc/api/v2/transform/nuc_json'
            . '?TAXID=' . $taxid
            . '&USERNAME=' . $username
            . '&FORMAT=JSON';


        $ch = curl_init($url);

        $body = json_encode($json, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

if ($body === false) {
    log_message('error', 'ERROR JSON_ENCODE: ' . json_last_error_msg());
    return ['error' => 'JSON encode failed'];
}

$body = mb_convert_encoding($body, 'UTF-8', 'UTF-8');

log_message('error', 'BODY ENVIADO FEL: ' . $body);

curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_SSL_VERIFYPEER => false,
    CURLOPT_SSL_VERIFYHOST => false,   // 🔥 FALTABA
    CURLOPT_CONNECTTIMEOUT => 30,      // 🔥 NUEVO
    CURLOPT_TIMEOUT => 60,             // 🔥 NUEVO
    CURLOPT_ENCODING => 'UTF-8',
    CURLOPT_HTTPHEADER => [
        "Authorization: Bearer " . $token,
        "Content-Type: application/json; charset=utf-8",
        "Accept: application/json",
        "Content-Length: " . strlen($body)
    ],
    CURLOPT_POSTFIELDS => $body,
]);


        log_message('error', 'JSON FEL: ' . json_encode($json, JSON_PRETTY_PRINT));

       $result = curl_exec($ch);

if ($result === false) {
    $error = curl_error($ch);
    $errno = curl_errno($ch);
    log_message('error', 'CURL ERROR FEL: ' . $errno . ' - ' . $error);
}

$http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
log_message('error', 'HTTP CODE FEL: ' . $http_code);

curl_close($ch);

log_message('error', 'FEL RESPONSE: ' . $result);

return json_decode($result, true);

    }

private function buildJsonDigifact($sale, $items, $cliente, $biller, $tipo)
{
    $detalle = [];
    $total_impuestos = 0;
    $gran_total = 0;

    foreach ($items as $indice => $i) {
        $item_precio_unitario = floatval($i['unit_price']);
$cantidad = floatval($i['quantity']);

$item_total = round($item_precio_unitario * $cantidad, 6);

// 🔥 Si el precio ya incluye IVA:
$taxable_amount = round($item_total / 1.12, 6);
$item_impuestos = round($taxable_amount * 0.12, 6);

        $detalle[] = [
            "NumeroLinea" => strval($indice + 1),
            "Number" => strval($indice + 1),
            "Codes" => null,
            "Type" => "Bien",
            "Description" => $i['product_name'],
            "Qty" => number_format($i['quantity'], 4, '.', ''),
            "UnitOfMeasure" => "UND",
            "Price" => number_format($item_precio_unitario, 5, '.', ''),
            "Discounts" => null,
            "Taxes" => [
                "Tax" => [
                    [
                        "Code" => "1",
                        "Description" => "IVA",
                        "TaxableAmount" => number_format($taxable_amount, 6, '.', ''),
                        "Amount" => number_format($item_impuestos, 5, '.', '')
        
                    ]
                ]
            ],
            "Totals" => [
                "TotalItem" => number_format($item_total, 4, '.', '')
            ]
        ];

        $total_impuestos += $item_impuestos;
        $gran_total += $item_total;
    }

    $cantidad_letras = ucwords(strtolower(str_replace(',', '', numfmt_format_currency(numfmt_create('es_GT', NumberFormatter::SPELLOUT), $gran_total, 'GTQ')))) . " CON 00/100";


    $json = [
        "Version" => "1.00",
        "CountryCode" => "GT",
        "Header" => [
            "DocType" => $tipo,
            "IssuedDateTime" => date('Y-m-d\TH:i:s-06:00', strtotime($sale['date'])),
            "Currency" => "GTQ"
        ],
        "Seller" => [
            "TaxID" => $biller['vat_no'],
            "TaxIDAdditionalInfo" => [
                [
                    "Name" => "AfiliacionIVA",
                    "Data" => null,
                    "Value" => "GEN"
                ]
            ],
            "Name" => $biller['name'],
            "AdditionalInfo" => [
                [
                    "Name" => "TipoFrase",
                    "Data" => "1",
                    "Value" => "1"
                ],
                [
                    "Name" => "Escenario",
                    "Data" => "1",
                    "Value" => "1"
                ],
                [
                    "Name" => "TipoFrase",
                    "Data" => "2",
                    "Value" => "2"
                ],
                [
                    "Name" => "Escenario",
                    "Data" => "2",
                    "Value" => "1"
                ]
            ],
            "BranchInfo" => [
                "Code" => "1",
                "Name" => $biller['name'],
                "AddressInfo" => [
                    "Address" => $biller['address'],
                    "City" => "01001",
                    "District" => $biller['city'],
                    "State" => $biller['state'],
                    "Country" => "GT"
                ]
            ]
        ],
        "Buyer" => [
            "TaxID" => $cliente['vat_no'], 
            "Name" => $cliente['name'], 
            "AddressInfo" => [
                "Address" => $cliente['address'],
                "City" => $cliente['postal_code'],
                "District" => $cliente['municipio'],
                "State" => $cliente['departamento'],
                "Country" => "GT"
            ]
        ],
        "Items" => $detalle,
        "Totals" => [
            "TotalTaxes" => [
                "TotalTax" => [
                    [
                        "Description" => "IVA",
                        "Amount" => number_format($total_impuestos, 5, '.', '')
                    ]
                ]
            ],
            "GrandTotal" => [
                "InvoiceTotal" => number_format($gran_total, 4, '.', '')
            ]
        ],
        "AdditionalDocumentInfo" => [
        "AdditionalInfo" => [
        [
            "Code" => "OBS",
            "Type" => "ADENDA",
            "AdditionalData" => [
                "Data" => [
                    [
                        "Name" => "DetallesAux_Detalle",
                        "Info" => [
                            [
                                "Name" => "NumeroLinea",
                                "Value" => "1"
                            ],
                            [
                                "Name" => "OBSERVACIONES",
                                "Value" => $sale['note'] ?? "-"
                            ],
                            [
                                "Name" => "CANTIDAD_LETRAS",
                                "Value" => $cantidad_letras
                            ]
                        ]
                    ]
                         ]
                                  ]
        ]
                         ]
                                        ]
];
array_walk_recursive($json, function (&$item) {
    if (is_string($item)) {
        $item = mb_convert_encoding($item, 'UTF-8', 'UTF-8');
    }
});

    return $json;
}
     private function saveInvoice($sale, $cliente, $res, $tipo, $biller)
{
    $this->db->insert('sma_fel_invoices', [
        'ticket_id'          => $sale['reference_no'],
        'client_nit'         => $cliente['vat_no'],
        'receptor_nombre'    => $cliente['name'],
        'receptor_direccion' => $cliente['address'],
        'receptor_municipio' => $cliente['municipio'],
        'receptor_departamento' => $cliente['departamento'],
        'receptor_pais'      => $cliente['pais'],
        'uuid'               => $res['uuid'] ?? null,
        'tipo_dte'           => $tipo,
        'created_at'         => date('Y-m-d H:i:s'),
        'auth_number'        => $res['authNumber'] ?? null,
        'xml_base64'         => $res['responseData1'] ?? null,
        'html_base64'        => $res['responseData2'] ?? null,
        'pdf_base64'         => $res['responseData3'] ?? null,
        'issued_at'          => $res['issuedTimeStamp'] ?? null,
        'serie'              => $res['serial'] ?? null,
        'numero'             => $res['numero'] ?? null,
        'company_id'         => $biller['id'] ?? null,
    ]);
}

private function logJson($json) {
    log_message('error', 'JSON FEL (DEBUG): ' . json_encode($json, JSON_PRETTY_PRINT));
}
}