<?php
defined('BASEPATH') or exit('No direct script access allowed');

class Merma extends MY_Controller
{
    public function __construct()
    {
        parent::__construct();

        if (!$this->loggedIn) {
            $this->session->set_userdata('requested_page', $this->uri->uri_string());
            $this->sma->md('login');
        }

        $this->load->library(['form_validation', 'datatables']);
        $this->load->model('admin/Merma_model'); // Carga el modelo de merma
        $this->load->model('admin/Products_model'); // Para buscar productos
        $this->load->model('admin/Sales_model'); // Para buscar tickets/ventas
    }

    /**
     * Listado de mermas
     */
    public function index()
    {
        $this->sma->checkPermissions();

        $bc = [['link' => base_url(), 'page' => lang('home')], ['link' => '#', 'page' => lang('mermas')]];
        $meta = ['page_title' => lang('mermas'), 'bc' => $bc];
        $this->page_construct('merma/index', $meta, $this->data);
    }

    /**
     * Formulario para agregar merma
     */
    public function add()
    {
        $this->sma->checkPermissions(false, true);

        // Si es POST, procesar la creación de merma
        if ($this->input->method() === 'post') {
            $ticket_ref = $this->input->post('ticket_ref');
            $cliente    = $this->input->post('cliente');
            $sucursal   = $this->input->post('sucursal_id');
            $nota       = $this->input->post('nota');

            $productos = $this->input->post('productos'); // Array de productos con cantidad y costo

            if (!$ticket_ref || !$productos) {
                $this->session->set_flashdata('error', 'Debe ingresar ticket y al menos un producto.');
                redirect($_SERVER['HTTP_REFERER']);
            }

            $data_merma = [];
            $now = date('Y-m-d H:i:s');
            $user = $this->session->userdata('user_id');

            foreach ($productos as $p) {
                $data_merma[] = [
                    'ticket_ref'     => $ticket_ref,
                    'cliente'        => $cliente,
                    'sucursal_id'    => $sucursal,
                    'producto_id'    => $p['id'],
                    'producto_codigo'=> $p['codigo'],
                    'cantidad'       => $p['cantidad'],
                    'costo_oculto'   => $p['costo'],
                    'nota'           => $nota,
                    'created_by'     => $user,
                    'created_at'     => $now,
                    'restaurado'     => 0
                ];

                // Descontar del inventario de la sucursal
                $this->Merma_model->descontarInventario($p['id'], $p['cantidad'], $sucursal);
            }

            if ($this->Merma_model->addMermaBatch($data_merma)) {
                $this->session->set_flashdata('message', 'Merma registrada correctamente');
                redirect('merma');
            } else {
                $this->session->set_flashdata('error', 'Error al guardar la merma');
                redirect($_SERVER['HTTP_REFERER']);
            }
        }

        // Cargar la vista del formulario
        $this->load->view($this->theme . 'merma/Add', $this->data);
    }

    /**
     * Buscar ticket por referencia
     */
    public function buscar_ticket()
    {
        $ref = $this->input->get('ref');
        if (!$ref) {
            echo json_encode([]);
            return;
        }

        $ticket = $this->Sales_model->getSaleByReference($ref);
        if (!$ticket) {
            echo json_encode([]);
            return;
        }

        echo json_encode([
            'ticket_ref' => $ticket->reference_no,
            'cliente'    => $ticket->customer_id,
            'sucursal_id'=> $ticket->warehouse_id
        ]);
    }

    /**
     * Buscar productos (autocomplete)
     */
    public function buscar_producto()
    {
        $term = $this->input->get('term');
        if (!$term) {
            echo json_encode([]);
            return;
        }

        $productos = $this->Products_model->searchProducts($term);
        $result = [];
        foreach ($productos as $p) {
            $result[] = [
                'id'      => $p->id,
                'codigo'  => $p->code,
                'nombre'  => $p->name,
                'costo'   => $p->cost
            ];
        }

        echo json_encode($result);
    }

    /**
     * Retornar merma al inventario
     */
    public function retornar($id)
    {
        $merma = $this->Merma_model->getMermaByID($id);
        if (!$merma || $merma->restaurado) {
            $this->session->set_flashdata('error', 'Merma no válida o ya retornada');
            redirect('merma');
        }

        $user = $this->session->userdata('user_id');
        $now  = date('Y-m-d H:i:s');

        // Retornar al inventario
        $this->Merma_model->retornarInventario($merma->producto_id, $merma->cantidad, $merma->sucursal_id);

        // Actualizar registro
        $this->Merma_model->updateMerma($id, [
            'restaurado' => 1,
            'updated_by' => $user,
            'updated_at' => $now
        ]);

        $this->session->set_flashdata('message', 'Merma retornada al inventario');
        redirect('merma');
    }

    /**
     * Desechar merma (sin retornar al inventario)
     */
    public function desechar($id)
    {
        $merma = $this->Merma_model->getMermaByID($id);
        if (!$merma || $merma->restaurado) {
            $this->session->set_flashdata('error', 'Merma no válida o ya procesada');
            redirect('merma');
        }

        $user = $this->session->userdata('user_id');
        $now  = date('Y-m-d H:i:s');

        // Solo actualizar como desecha (restaurado = 0, pero se guarda el historial)
        $this->Merma_model->updateMerma($id, [
            'updated_by' => $user,
            'updated_at' => $now
        ]);

        $this->session->set_flashdata('message', 'Merma desecha registrada');
        redirect('merma');
    }
}
